/* 
 *  ClientData.java
 * 
 *  Copyright 2016 Avaya Inc. All Rights Reserved.
 * 
 *  Usage of this source is bound to the terms described
 *  in AvayaLicenseSDK.rtf.
 * 
 *  Avaya - Confidential & Proprietary. Use pursuant to your signed agreement
 *  or Avaya Policy
 * 
 */
package com.avaya.ccs.javafxrefclient;

import com.avaya.ccs.api.ClientI;
import com.avaya.ccs.api.enums.ClientState;
import java.util.LinkedList;
import java.util.List;
import javafx.util.Pair;

/**
 * This class is a holder for the client related state. It is used to send
 * copies of relevant client data to the GUI thread. It is then up to the GUI
 * thread to decide what and how to update on the GUI.
 */
public class ClientData {

    private final boolean connected;
    private boolean autoReconnect = true;
    private ClientState state = ClientState.DISCONNECTED;
    private boolean ha = false;
    private boolean haOperational = false;
    private String server = "";
    private String sdkVersion = "";
    private String connectedServer = "";

    public ClientData(ClientI client) {
        if (client != null) {
            connected = client.isConnected();
            autoReconnect = client.getAutoReconnect();
            state = client.getState();
            ha = client.isHa();
            haOperational = client.isHaOperational();
            server = client.getServer();
            sdkVersion = client.getVersion();
            connectedServer = client.getConnectedServer();
        } else {
            connected = false;
        }
    }

    public boolean isConnected() {
        return this.connected;
    }

    @Override
    public String toString() {
        return " connected:" + connected + " autoReconnect:" + autoReconnect + " state:" + state.name() + " HA:" + isHa() + " haOperational:" + isHaOperational();
    }

    public boolean isAutoReconnect() {
        return autoReconnect;
    }

    public ClientState getState() {
        return state;
    }

    public boolean isHa() {
        return ha;
    }

    public boolean isHaOperational() {
        return haOperational;
    }

    public List<Pair<String, String>> getClientProperties() {
        List<Pair<String, String>> clientProperties = new LinkedList<>();
        clientProperties.add(new Pair<String, String>("Connected", String.valueOf(this.isConnected())));
        clientProperties.add(new Pair<String, String>("Auto Reconnect", String.valueOf(this.isAutoReconnect())));
        clientProperties.add(new Pair<String, String>("HA", String.valueOf(this.isHa())));
        clientProperties.add(new Pair<String, String>("HA Operational", String.valueOf(this.isHaOperational())));
        clientProperties.add(new Pair<String, String>("State", this.getState().name()));
        clientProperties.add(new Pair<String, String>("Server", this.getServer()));
        clientProperties.add(new Pair<String, String>("SDK Version", this.getSdkVersion()));
        clientProperties.add(new Pair<String, String>("Connected Server", this.getConnectedServer()));
        return clientProperties;
    }

    public String getServer() {
        return server;
    }

    public String getSdkVersion() {
        return sdkVersion;
    }

    public String getConnectedServer() {
        return connectedServer;
    }
}
